
#include <dlfcn.h>
#include <sys/queue.h>
#include <dirent.h>
#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <pthread.h>
#include <limits.h>
#include <sys/resource.h>
#include <string.h>

#include "netXTransport.h"
#include "netxtransport_linux.h"
#include "TL_Marshaller.h"
#include "TL_rcXPacket.h"

#define MAX_FILE_NAME_LENGTH 256

#define CONNECTOR_INIT   "connector_init"
#define CONNECTOR_DEINIT "connector_deinit"

#ifndef CONNECTOR_BASE_PATH
#define CONNECTOR_BASE_PATH "./connectors/"
#endif

typedef int32_t(*PFN_CONNECTOR_INIT)(void);
typedef void(*PFN_CONNECTOR_DEINIT)(void);

typedef struct TRANSPORT_CONNECTOR_Ttag
{
  TAILQ_ENTRY(TRANSPORT_CONNECTOR_Ttag) tList;
  void*                                 hConnectorFile;
  PFN_CONNECTOR_INIT                    pfnConnectorInit;
  PFN_CONNECTOR_DEINIT                  pfnConnectorDeInit;
} TRANSPORT_CONNECTOR_T;

TAILQ_HEAD ( CONN_LIST, TRANSPORT_CONNECTOR_Ttag);

struct CONN_LIST s_tConnectorList;

static TL_INIT_T* s_ptDataLayerInit = NULL;

/*****************************************************************************/
/*! Load connector library
*   \param libPath    path to library to be loaded                           */
/*****************************************************************************/
void load_connector( char* libPath) {
  FILE*                hFile = NULL;
  PFN_CONNECTOR_INIT   pfnConnectorInit;
  PFN_CONNECTOR_DEINIT pfnConnectorDeInit;

  TAILQ_INIT(&s_tConnectorList);

  if (NULL == (hFile = dlopen( libPath, RTLD_NOW | RTLD_LOCAL))) {
    fprintf(stderr, "Error loading plugin library %s with error=%s)\n", libPath, dlerror());
    return;
  }
  if( (NULL == (pfnConnectorInit = dlsym(hFile, CONNECTOR_INIT))) ||
      (NULL == (pfnConnectorDeInit = dlsym(hFile, CONNECTOR_DEINIT))) ) {
    fprintf(stderr, "Error loading plugin library %s, as it does not contain required exports\n", libPath);
  } else {
    if (0 == pfnConnectorInit()) {
      TRANSPORT_CONNECTOR_T* connector = calloc(1, sizeof(*connector));

      if (connector != NULL) {
        connector->hConnectorFile     = hFile;
        connector->pfnConnectorInit   = pfnConnectorInit;
        connector->pfnConnectorDeInit = pfnConnectorDeInit;

        TAILQ_INSERT_HEAD(&s_tConnectorList, connector, tList);
        return;
      } else {
        fprintf(stderr, "Failed to allocate memory stopping connector!)\n");
        pfnConnectorDeInit();
      }
    }
  }
  dlclose(hFile);
}

/*****************************************************************************/
/*! Unload all previously loaded connector                                   */
/*****************************************************************************/
void unload_connector() {
  TRANSPORT_CONNECTOR_T* connector = NULL;

  while(NULL != (connector = TAILQ_FIRST((&s_tConnectorList))))
  {
    if (connector->pfnConnectorDeInit != NULL)
      connector->pfnConnectorDeInit();

    if (connector->hConnectorFile != NULL)
      dlclose(connector->hConnectorFile);

    TAILQ_REMOVE(&s_tConnectorList, connector, tList);
    free(connector);
  }
}

/*****************************************************************************/
/*! Iterates over folder given by szPath and tries to load connector lib
*   \param szPath    path to search for library                              */
/*****************************************************************************/
void search_connnector( char* szPath) {
  DIR* dir;

  /* Iterate over connectors in [currentdir]/connectors folder */
  if(NULL != (dir = opendir(szPath))) {
    struct dirent* dirent;

    while(NULL != (dirent = readdir(dir))) {
      char* szExt = strstr(dirent->d_name, ".");
      if(NULL != szExt) {
        if(0 == strncasecmp(szExt, ".so", 3)) {
          void* hFile = NULL;
          char libPath[MAX_FILE_NAME_LENGTH];

          snprintf( libPath, MAX_FILE_NAME_LENGTH, "%s/%s", szPath, dirent->d_name);
          load_connector( libPath);
        }
      }
    }
  }
}

/*****************************************************************************/
/*! Initializes the cifX Marshaller Translation-Layer and the rcX-Packet
*   Translation-Layer.
*   \param pvParam    User param (currently not used)
*   \return NXT_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t TLLayerInit( void* pvParam)
{
  int32_t lRet = NXT_NO_ERROR;

  if (NXT_NO_ERROR == (lRet = cifX_Marshaller_Init( pvParam)))
  {
    lRet = rcXPacket_Init( pvParam);
  }
  return lRet;
}

/*****************************************************************************/
/*! De-initializes translation layer
*   \param pvParam    User param (currently not used)                        */
/*****************************************************************************/
void TLLayerDeInit( void* pvParam)
{
  cifX_Marshaller_DeInit( pvParam);
  rcXPacket_DeInit( pvParam);
}

extern uint32_t g_ulTraceLevel;

/*****************************************************************************/
/*! Initializes netxtransport toolkit
*   \param pvParam    User param (currently not used)                        */
/*****************************************************************************/
int32_t cifXDriverInit( const struct CIFX_LINUX_INIT* init_params) {
  pthread_attr_t polling_thread_attr = {{0}};
  int32_t        lRet                = NXT_NO_ERROR;
  struct rlimit  rlim;

  if (init_params != NULL)
    g_ulTraceLevel = init_params->trace_level;

  if (init_params != NULL)
    g_ulTraceLevel = init_params->trace_level;

  if (s_ptDataLayerInit != NULL)
    return NXT_NO_ERROR;

  if (NULL == (s_ptDataLayerInit = malloc(sizeof(TL_INIT_T))))
    return NXT_NO_MEMORY;

  /* setup netXTransport initialization structure */
  s_ptDataLayerInit->pfnTLInit   = TLLayerInit;   /* function pointer the Translation-Layer initialization    */
  s_ptDataLayerInit->pfnTLDeInit = TLLayerDeInit; /* function pointer the Translation-Layer de-initialization */
  s_ptDataLayerInit->pvData      = NULL;          /* Private data (currently  not used)                       */

  getrlimit( RLIMIT_CORE, &rlim);

  rlim.rlim_cur = -1;
  setrlimit( RLIMIT_CORE, &rlim);

  if (NXT_NO_ERROR == (lRet = netXTransportInit( s_ptDataLayerInit, sizeof(*s_ptDataLayerInit)))) {
    char szPath[CIFX_MAX_FILE_NAME_LENGTH];

    snprintf(szPath, MAX_FILE_NAME_LENGTH, CONNECTOR_BASE_PATH);
    /* check for connectors */
    search_connnector( szPath);
    /* start netXTransport (starts device discovering and registration) */
    if (NXT_NO_ERROR != (lRet = netXTransportStart( NULL, NULL))) {
      /* de-initialize toolkit */
      netXTransportDeinit();
      unload_connector();
    }
  }
  if (s_ptDataLayerInit != NULL)
    free(s_ptDataLayerInit);

  s_ptDataLayerInit = NULL;
  return lRet;
}

/*****************************************************************************/
/*! De-initializes netxtransport toolkit
*   \param pvParam    User param (currently not used)                        */
/*****************************************************************************/
void cifXDriverDeinit() {

  if (s_ptDataLayerInit != NULL) {
    netXTransportDeinit();
    unload_connector();
    free( s_ptDataLayerInit);
    s_ptDataLayerInit = NULL;
  }
}
